import os
import uuid
import types
from dataclasses import asdict, dataclass
from typing import Any, DefaultDict, Dict, List, Optional, Tuple

import bullet_safety_gym  # noqa
import dsrl
import gymnasium as gym  # noqa
import gym as gym_org
import numpy as np
import pyrallis
import torch
from dsrl.infos import DENSITY_CFG
from dsrl.offline_env import OfflineEnvWrapper, wrap_env  # noqa
from fsrl.utils import WandbLogger
from fsrl.utils import TensorboardLogger
from torch.utils.data import DataLoader
from tqdm.auto import trange  # noqa

from examples.configs.rtg_configs import RTG_DEFAULT_CONFIG, RTGTrainConfig
from osrl.algorithms import State_AE, Action_AE, inverse_dynamics_model, ActionAETrainer, StateAETrainer
from osrl.algorithms import RTG_model, RTGTrainer
from osrl.common import SequenceDataset
from osrl.common.exp_util import auto_name, seed_all, load_config_and_model


@pyrallis.wrap()
def train(args: RTGTrainConfig):
    # update config
    cfg, old_cfg = asdict(args), asdict(RTGTrainConfig())
    differing_values = {key: cfg[key] for key in cfg.keys() if cfg[key] != old_cfg[key]}
    cfg = asdict(RTG_DEFAULT_CONFIG[args.task]())
    cfg.update(differing_values)
    args = types.SimpleNamespace(**cfg)

    # setup logger
    default_cfg = asdict(RTG_DEFAULT_CONFIG[args.task]())
    if args.name is None:
        args.name = auto_name(default_cfg, cfg, args.prefix, args.suffix)
    
    if args.use_state:
        args.name = args.name+"_posterior"
    else:
        args.name = args.name+"_prior"
    if args.group is None:
        args.group = args.task + "-cost-" + str(int(args.cost_limit))
    if args.logdir is not None:
        args.logdir = os.path.join(args.logdir, args.group, args.name)
    # logger = WandbLogger(cfg, args.project, args.group, args.name, args.logdir)
    logger = TensorboardLogger(args.logdir, log_txt=True, name=args.name)
    logger.save_config(cfg, verbose=args.verbose)

    # set seed
    seed_all(args.seed)
    if args.device == "cpu":
        torch.set_num_threads(args.threads)

    # initialize environment
    if "Metadrive" in args.task:
        # import gym
        env = gym_org.make(args.task)
    else:
        env = gym.make(args.task)

    # pre-process offline dataset
    data = env.get_dataset()
    env.set_target_cost(args.cost_limit)

    cbins, rbins, max_npb, min_npb = None, None, None, None
    if args.density != 1.0:
        density_cfg = DENSITY_CFG[args.task + "_density" + str(args.density)]
        cbins = density_cfg["cbins"]
        rbins = density_cfg["rbins"]
        max_npb = density_cfg["max_npb"]
        min_npb = density_cfg["min_npb"]
    data = env.pre_process_data(data,
                                args.outliers_percent,
                                args.noise_scale,
                                args.inpaint_ranges,
                                args.epsilon,
                                args.density,
                                cbins=cbins,
                                rbins=rbins,
                                max_npb=max_npb,
                                min_npb=min_npb)

    # wrapper
    env = wrap_env(
        env=env,
        reward_scale=args.reward_scale,
    )
    env = OfflineEnvWrapper(env)

    state_encoder = None
    action_encoder = None

    # model & optimizer & scheduler setup
    state_dim = env.observation_space.shape[0]
    action_dim = env.action_space.shape[0]
    # if state_encoder is not None:
    #     state_dim = senc_cfg["state_encode_dim"]
    # if action_encoder is not None:
    #     action_dim = aenc_cfg["action_encode_dim"]
    model = RTG_model(
        state_dim=state_dim,
        prompt_dim=args.prompt_dim,
        cost_embedding_dim=args.embedding_dim,
        state_embedding_dim=args.embedding_dim,
        prompt_embedding_dim=args.embedding_dim,
        r_hidden_sizes=args.r_hidden_sizes,
        use_state=args.use_state,
        use_prompt=args.use_prompt
    ).to(args.device)
    print(f"Total parameters: {sum(p.numel() for p in model.parameters())}")

    def checkpoint_fn():
        return {"model_state": model.state_dict()}

    logger.setup_checkpoint_fn(checkpoint_fn)

    # trainer
    trainer = RTGTrainer(model,
                         logger=logger,
                         learning_rate=args.learning_rate,
                         device=args.device,
                         logprob_loss=args.logprob_loss)

    ct = lambda x: 70 - x if args.linear else 1 / (x + 10)

    dataset = SequenceDataset(
        data,
        seq_len=1,
        reward_scale=args.reward_scale,
        cost_scale=args.cost_scale,
        deg=args.deg,
        pf_sample=args.pf_sample,
        max_rew_decrease=args.max_rew_decrease,
        beta=args.beta,
        augment_percent=0.0,
        cost_reverse=args.cost_reverse,
        max_reward=args.max_reward,
        min_reward=args.min_reward,
        pf_only=args.pf_only,
        rmin=args.rmin,
        cost_bins=args.cost_bins,
        npb=args.npb,
        cost_sample=False,
        cost_transform=None,
        start_sampling=args.start_sampling,
        prob=args.prob,
        random_aug=0.0,
        aug_rmin=args.aug_rmin,
        aug_rmax=args.aug_rmax,
        aug_cmin=args.aug_cmin,
        aug_cmax=args.aug_cmax,
        cgap=args.cgap,
        rstd=args.rstd,
        cstd=args.cstd,
        state_encoder=state_encoder,
        action_encoder=action_encoder,
        device=args.device
    )

    trainloader = DataLoader(
        dataset,
        batch_size=args.batch_size,
        pin_memory=True,
        num_workers=args.num_workers,
    )
    trainloader_iter = iter(trainloader)

    testloader = DataLoader(
        dataset,
        batch_size=args.batch_size*100,
        pin_memory=True,
        num_workers=args.num_workers,
    )
    testloader_iter = iter(testloader)

    # ready for modification
    scale = 1.0
    if args.need_rescale:
        scale = 0.01
    best_loss = np.inf
    for epoch in range(args.epoch_num):
        for step in trange(args.steps_per_epoch, desc="Training"):
            batch = next(trainloader_iter)
            states, actions, returns, costs_return, time_steps, mask, episode_cost, costs = [
                b.to(args.device) for b in batch
            ]
        # for i, (states, actions, returns, costs_return, time_steps, mask, episode_cost, costs) in enumerate(trainloader):
        #     states = states.to(args.device)
        #     returns = returns.to(args.device).unsqueeze(-1)
        #     costs_return = costs_return.to(args.device)
            returns = returns.unsqueeze(-1)
            if args.use_state:
                trainer.train_one_step(returns * scale, costs_return, states=states)
            else:
                trainer.train_one_step(returns * scale, costs_return)
        logger.save_checkpoint()
        test_batch = next(testloader_iter)
        states, actions, returns, costs_return, time_steps, mask, episode_cost, costs = [
            b.to(args.device) for b in test_batch
        ]
        returns = returns.unsqueeze(-1)
        if args.use_state:
            eval_rtg_loss = trainer.eval_one_step(returns * scale, costs_return, states=states)
        else:
            eval_rtg_loss = trainer.eval_one_step(returns * scale, costs_return)
        if eval_rtg_loss < best_loss:
            best_loss = eval_rtg_loss
            best_idx = epoch+1
            logger.save_checkpoint(suffix="best")
        logger.store(tab="train", best_idx=best_idx)
        logger.write(epoch+1, display=False)

    # for saving the best
    # best_reward = -np.inf
    # best_cost = np.inf
    # best_idx = 0

    # for step in trange(args.update_steps, desc="Training"):
    #     batch = next(trainloader_iter)
    #     states, actions, returns, costs_return, time_steps, mask, episode_cost, costs = [
    #         b.to(args.device) for b in batch
    #     ]
    #     if args.use_state:
    #         trainer.train_one_step
    #     trainer.train_one_step(states, actions, returns, costs_return, time_steps, mask,
    #                            episode_cost, costs)



if __name__ == "__main__":
    train()
